//  fastlowerbound.cpp
//  Author: Yun S. Song
//  October 2004

#include <iostream>
#include "fastlowerbound.h"
#include "basicdefs.h"

// test incompatibility between columns i and j
bool testIncomp(const BinMatrix & inData, const unsigned int iN, const unsigned int jN){
  unsigned int irow=0;
  bool test[2][2]={{0,0},{0,0}};
  for(irow=0; irow < inData.rowsize(); irow++){
    test[inData[irow][iN]][inData[irow][jN]]=1;
  }
  return (test[0][0]*test[1][0]*test[0][1]*test[1][1]);
}


// find incompatibility matrix
USMatrix buildIncompMatrix(const BinMatrix &inData){
  USMatrix incomp(inData.colsize());
  unsigned int iN,jN;
  if(inData.colsize()!=0){
    for(iN=0; iN < incomp.size()-1; iN++){
      for(jN=iN+1; jN < incomp.size(); jN++){
	incomp.mat[iN][jN]=testIncomp(inData,iN,jN);
	incomp.mat[jN][iN]=incomp[iN][jN];
      }
    }
  }
  return incomp;
}


// Myers & Griffiths Algorithm 1
// Given a matrix of local bounds, this function find a global bound for the 
// the entire data.
unsigned int globalBound(const USMatrix &localBoundMat){
  unsigned int icol,intermediate;
  // contains bounds between the first column and column K
  vector<unsigned int> boundsOne2K(localBoundMat.size()); 
  unsigned int tmpmax, tmpsum;
  //R_{0,0} = 0
  if(localBoundMat.size()!=0){
    boundsOne2K[0]=0;
    // Dynamic programming in algorithm 1
    for(icol=1; icol < localBoundMat.size(); icol++){
      tmpmax=0;
      for(intermediate=0; intermediate < icol; intermediate++){
	tmpsum=boundsOne2K[intermediate]+localBoundMat[intermediate][icol];
	if(tmpsum > tmpmax) tmpmax=tmpsum;
      }
      boundsOne2K[icol]=tmpmax;
    }
    return boundsOne2K[localBoundMat.size()-1];
  }
  return 0;
}
      

// This routine will modify incomp by using #distint hap - #distinct col - 1 bounds
// First compute bounds for intervals of size 1, then size 2, and so on.
// For the local bound of a, use max(dRdC(a), dRdC(b), dRdC(c))
// |----------|  a
// |---------|   b
//  |---------|  c
void dRdC_updateIncMatrix(USMatrix &incomp, const BinMatrix &inData){
  unsigned int itvSize; // interval size
  unsigned int head, tail;
  // No need to consider itvSize=1
  // If there is incompatibility, then dRow_dCol=1.  
  // If there is no incompatibility, then dRow_dCol=0.
  // So, we can just use the incompatibility matrix for itvSize=1.
  for(itvSize=2; itvSize < incomp.size(); itvSize++){
    for(head=0; head < incomp.size()-itvSize; head++){
      tail=head+itvSize;
      // Reduce the submatrix before finding the C(n,2) bound
      // This actually makes difference.  
      // For Kreitman's data, reducing first gives 6 as the C(n,2) global bound
      // If reducing is not used, the C(n,2) global bound is 5.
      incomp.mat[head][tail]=maxOf4(&(incomp[head][tail-1]), &(incomp[head+1][tail]),&(incomp[head][tail]), dRow_dCol(getReducedSubMat(inData,head,tail)));
      //incomp.mat[head][tail]=max(max(max(incomp[head][tail-1], incomp[head+1][tail]), dRow_dCol(getReducedSubMat(inData,head,tail))),incomp[head][tail]);

    }
  }
}

unsigned int fastlowerbound(const BinMatrix &data, short method){
  //  extern  short flbMethod; // from main-upper.cpp
  USMatrix localBounds;
  unsigned int result;
  if(data.rowsize() > 3){
    // find incompatibility in the remaining data 
    localBounds=buildIncompMatrix(data);
  
    if(method==2){  // C(n,2) haplotype lower bound
      dRdC_updateIncMatrix(localBounds,data);
    }
    
    result=globalBound(localBounds);
    localBounds.clear();

    return(result);
  }
  else return 0;
}
