// C++ code
// basicdefs.cpp
// Author: Yun S. Song
// October 2004

#include <iostream>
#include <vector>
#include <math.h>
#include "basicdefs.h"
#include <stdio.h>
#include <ctype.h>
#include <fstream>
// description of the program
void progdesc(){
  printf("SYNOPSIS: shrub [-d #] [-l #] [-a [#]] [-r #] [-g name] data-filename\n\n");
  printf("OPTIONS: (See README-SHRUB for a more detailed description of the program.) \n");
  printf("  -d num    Display setting.\n");
  printf("            0: Only display final results (default). \n");
  printf("            1: Display the original and the reduced data.\n");
  printf("            2: Display as in 1 + display local bound matrices for global\n");
  printf("               lower bounds.\n\n");
  printf("  -l num    Fast lower bound method to be used in branch and bound.\n");
  printf("            1: Hudson and Kaplan's bound (default).\n");
  printf("            2: Approximate haplotype bound using distinct columns and rows.\n\n");
  printf("  -a [num]  Obtain only a fast upper bound.\n");
  printf("            num = the maximum number of row removals to be tried for every\n");
  printf("            matrix in step 3' of the algorithm.  If a number is not specified,\n");
  printf("            then all rows satisfying the condition in step 3' will be tried.\n\n");
  printf("  -r num    Number of '-a 1' runs to be executed. (default is 1)\n\n");
  printf("  -g name   Output an ARG with the computed number of recombination\n");
  printf("            events into a file.  The output is in GML format.\n\n");
  //  printf("See README-SHRUB for a more detailed description of the program.\n\n");
  exit(1);
}

// get options from command line
void getoptions(int argc, char *argv[], char **gfile, char **fname){
  extern short displayData;
  extern short flbMethod;
  extern short displayARG;
  extern bool  approximate;
  extern unsigned short maxRows2Look;
  extern unsigned int numberOfRuns;
  if(argc < 2){
    progdesc();
  }
  else{
    *fname=argv[argc-1];
    for(int iI=1; iI < argc-1; iI++){
      if(argv[iI][0] != '-'){
	cerr << "ERROR: Options must begin with '-'.\n\n";
	//progdesc();
	exit(1);
      }
      else{	  
	switch(argv[iI][1]){
	case 'a':{
	  approximate=true;
	  if(iI+1 != argc - 1 && argv[iI+1][0] != '-'){
	    if(atoi(argv[iI+1]) < 1){
	      cerr << "ERROR: The maximum number of row removals must be a positive integer.\n";
	      exit (1);
	    }
	    else{
	      maxRows2Look=(unsigned short) atoi(argv[iI+1]);
	      iI++;
	    }
	  }
	  break;
	}
	case 'r':{
	  if(atoi(argv[iI+1]) < 1){
	    cerr << "ERROR: Numer of runs must be a positive integer.\n";
	    exit (1);
	  }
	  else{	    
	    numberOfRuns=(unsigned int)atoi(argv[iI+1]);
	    iI++;
	  }
	  break;
	}
	case 'd':{
	  displayData=atoi(argv[iI+1]);
	  if(displayData < 0 || displayData > 2){
	    cerr << "ERROR: Invalid display option.\n";
	    // progdesc();
	    exit (1);
	  }
	  iI++;
	  break;
	}
	case 'l':{
	  flbMethod=atoi(argv[iI+1]);
	  if(flbMethod != 1 && flbMethod != 2){
	    cerr << "ERROR: Invalid fast lower bound method is chosen.\n";
	    cerr << "       Choose either 1 or 2 with the '-l' option.\n\n";
	    //progdesc();
	    exit (1);
	  }
	  iI++;
	  break;
	}
	case 'g':{
	  displayARG=1;
	  *gfile=argv[iI+1];
	  if(iI+1 == argc-1 || argv[iI+1][0] == '-'){
	    cerr << "ERROR: Need to specify file name after the '-g' option.\n\n";
	    //progdesc();
	    exit (1);
	  }
	  iI++;
	  break;
	}
	default:
	  printf("ERROR: Unknown argument type.\n\n");
	  //progdesc();
	  exit(1);
	  break;
	}
      }
    }
  }
}
	
unsigned short max(const unsigned short one,const unsigned  short two){
  return( (one >= two)? one: two);
}

unsigned short maxOf4(const unsigned short *one, const unsigned short *two,const unsigned short *three, unsigned short four){
  unsigned short tmp=0;
  if(*one > tmp) tmp=*one;
  if(*two > tmp) tmp=*two;
  if(*three > tmp) tmp=*three;
  if(four > tmp) tmp=four;
  return tmp;
}

void newline(){
  cout << endl;
}

void smallspace(){
  cout << " ";
}

void removeEntry(const unsigned int entry, vector<unsigned int> &vec){
  unsigned int iR=0;
  for(iR=entry; iR < vec.size()-1; iR++){
    vec[iR]=vec[iR+1];
  }
  vec.resize(vec.size()-1);
}
    


// conjugate a columns, i.e. 0 -> 1 and 1 -> 0
std::vector<bool> conjugate(const std::vector<bool> &vec)
{
  std::vector<bool> conjVec( vec.size() );
  unsigned int iC;
  for(iC = 0; iC < vec.size(); iC++){
    conjVec[iC] = (vec[iC]+1)%2;
  }
  return(conjVec);
}


// generates a vector of intergers from begin value to end value, inclusive
vector<unsigned int> table(unsigned int begin, unsigned int end)
{
  std::vector<unsigned int> outvec(end-begin+1);
  unsigned int iM=0, iN;
  for(iN=begin; iN <= end; iN++){
    outvec[iM]=iN;
    iM++;
  }
  return(outvec);
}

// generates a vector of intergers from begin value to end value, inclusive
vector<unsigned short> table(unsigned short begin, unsigned short end)
{
  std::vector<unsigned short> outvec(end-begin+1);
  unsigned short iM=0, iN;
  for(iN=begin; iN <= end; iN++){
    outvec[iM]=iN;
    iM++;
  }
  return(outvec);
}
    
// generates a vector of intergers from begin value to end value, with a chosen 
// intermediate interger removed
vector<unsigned short> brokenTable(unsigned short begin, unsigned short end, unsigned short toRemove )
{
  std::vector<unsigned short> outvec(end-begin);
  unsigned short iM=0, iN;
  for(iN=begin; iN < toRemove; iN++){
    outvec[iM]=iN;
    iM++;
  }
  for(iN=toRemove+1; iN <= end; iN++){
    outvec[iM]=iN;
    iM++;
  }
  return(outvec);
}


// test whether two haplotypes in the region between startcol and endcol are identical
bool sameHapQ(const BinMatrix &inMat, const unsigned int row1, const unsigned int row2, const unsigned int startCol, const unsigned int endCol){
  unsigned int icol;
  for(icol=startCol; icol <= endCol; icol++){
    if(inMat[row1][icol] != inMat[row2][icol]){
      return false;
    }
  }
  return true;
}

// inMat is a reduced matrix
// test whether two haplotypes in the region between startcol and endcol are identical
bool sameHapQ(const BinMatrix &inMat, const unsigned int row1, const unsigned int row2){
  unsigned int icol;
  for(icol=0; icol < inMat.colsize(); icol++){
    if(inMat[row1][icol] != inMat[row2][icol]){
      return false;
    }
  }
  return true;
}

// check whether 2 columns are equivalent
// 2 columns are equivalent if they are either identical or become identical after 0 <-> 1.
bool equivColQ(const BinMatrix &inMat,  unsigned short *col1,  unsigned short *col2){
  unsigned int irow;
  bool testresult=true;
  for(irow=0; irow < inMat.rowsize(); irow++){
    if(inMat[irow][*col1] != inMat[irow][*col2]){
      testresult=false;
      break;
    }
  }
  if(testresult == false){ // if the above test returns false
    testresult=true;       // check to see if the two columns are conjugate
    for(irow=0; irow < inMat.rowsize(); irow++){
      if(inMat[irow][*col1] != (inMat[irow][*col2]+1)%2){ // mod 2
	testresult=false;
	break;
      }
    }
  }
  return testresult;
}

// inMat is a reduced matrix
// count the number of unique haplotypes for the region between start column and end column
unsigned int countUniqueRows(const BinMatrix &inMat){

  vector<unsigned int> distinct(inMat.rowsize());
  unsigned int iR;
  std::vector<vector<unsigned short> > RemainingTest(2);
  RemainingTest[0].resize(inMat.rowsize());   RemainingTest[1].resize(inMat.rowsize());
  unsigned short distinctSize=0, remainingSize=0, needToTestSize=inMat.rowsize();
  bool parity=0, newparity;

  RemainingTest[0]=table(0, (unsigned short) (inMat.rowsize()-1) );
  while(needToTestSize!=0){
    distinct[distinctSize]=RemainingTest[parity][0];
    distinctSize++;
    newparity=(parity+1)%2;
    remainingSize=0;
    for(iR=1; iR < needToTestSize; iR++){
      // not the same as row=needToTest[0] 
      if(inMat[RemainingTest[parity][iR]]!=inMat[RemainingTest[parity][0]]){ 
      //      if(sameHapQ(inMat,RemainingTest[parity][iR],RemainingTest[parity][0])==false){ 
	RemainingTest[newparity][remainingSize]=RemainingTest[parity][iR];
	remainingSize++;
      }
    }
    needToTestSize=remainingSize;
    parity=newparity;
  }
  RemainingTest.clear();
  distinct.clear();
  return(distinctSize);
}


// inMat is a reduced matrix
// count the number of inequivalent columns for the region between start column and end column
unsigned int countUniqueCols(const BinMatrix &inMat){

  vector<unsigned int> distinct(inMat.colsize());
  unsigned int iR;
  std::vector<vector<unsigned short> > RemainingTest(2);
  RemainingTest[0].resize(inMat.colsize());   RemainingTest[1].resize(inMat.colsize());
  unsigned short distinctSize=0, remainingSize=0, needToTestSize=inMat.colsize();
  bool parity=0, newparity;

  RemainingTest[0]=table(0, (unsigned short) (inMat.colsize()-1) );
  while(needToTestSize!=0){
    distinct[distinctSize]=RemainingTest[parity][0];
    distinctSize++;
    newparity=(parity+1)%2;
    remainingSize=0;
    for(iR=1; iR < needToTestSize; iR++){
      // not the same as col=needToTest[0] 
      if(equivColQ(inMat,&RemainingTest[parity][iR],&RemainingTest[parity][0])==false){
	RemainingTest[newparity][remainingSize]=RemainingTest[parity][iR];
	remainingSize++;
      }
    }
    needToTestSize=remainingSize;
    parity=newparity;
  }
  RemainingTest.clear();
  distinct.clear();
  return(distinctSize);
}


// inMat is reduced matrix
// find #distinct row - #distinct columns - 1
unsigned short dRow_dCol(const BinMatrix &inMat){
  int temp;
  if(inMat.rowsize() < 4){
    return 0;
  }
  else{
    temp=(int) countUniqueRows(inMat)-(int) countUniqueCols(inMat)-1;
    return (temp > 0) ? (unsigned short) temp: 0;
  }
}


// returns a new matrix with noninformative sites removed
BinMatrix killNonInf(const BinMatrix &inMat){
  BinMatrix tmpMat;
  unsigned int tmplength=inMat.rowsize()-1;
  unsigned int iCount=0; // counts the number of informative columns
  unsigned int tmpnorm;
  vector<unsigned int> informative(inMat.colsize());
  unsigned int i,j;
  if(inMat.colsize() > 0){
    for(unsigned int iC=0; iC < inMat.colsize(); iC++){
      tmpnorm=inMat.colnorm(iC);
      if(tmpnorm > 1 && tmpnorm < tmplength){ // informative
	informative[iCount]=iC;
	iCount++;
      }
    }

    tmpMat.resize(inMat.rowsize(),iCount);
    for(j=0; j < iCount; j++){
      for(i=0; i < inMat.rowsize(); i++){
	tmpMat.mat[i][j]=inMat[i][informative[j]];
      }
    }
    return(tmpMat);
  }
  else return(inMat);
}



// coalesce indentical sequences
// Used in computing fast lower bounds
BinMatrix coalesceFL(const BinMatrix &inMat){

  BinMatrix reducedMat;

  vector<unsigned short> distinct(inMat.rowsize());
  unsigned short iR;
  std::vector<vector<unsigned short> > RemainingTest(2);
  RemainingTest[0].resize(inMat.rowsize());   RemainingTest[1].resize(inMat.rowsize());
  unsigned short distinctSize=0, remainingSize=0, needToTestSize=inMat.rowsize();
  bool parity=0, newparity;

  if(inMat.rowsize() > 1){
    RemainingTest[0]=table(0,(unsigned short) (inMat.rowsize()-1) );
    while(needToTestSize!=0){
      distinct[distinctSize]=RemainingTest[parity][0];
      distinctSize++;
      newparity=(parity+1)%2;
      remainingSize=0;
      for(iR=1; iR < needToTestSize; iR++){
	// not the same as row=RemainingTest[parity][0]
	if(inMat[RemainingTest[parity][iR]]!=inMat[RemainingTest[parity][0]]){ 
	//	if(sameHapQ(inMat,RemainingTest[parity][iR],RemainingTest[parity][0])==false){ 
	  RemainingTest[newparity][remainingSize]=RemainingTest[parity][iR];
	  remainingSize++;
	}
      }
      needToTestSize=remainingSize;
      parity=newparity;
    }

    reducedMat.mat.resize(distinctSize);
    for(iR=0; iR < distinctSize; iR++){
      reducedMat.mat[iR]=inMat[distinct[iR]];
    }  
    RemainingTest.clear();
    distinct.clear();
    return(reducedMat);
  }
  else return(inMat);
}


// coalesce indentical sequences
BinMatrix coalesce(const BinMatrix &inMat, vector <unsigned short> &rmap){

  BinMatrix reducedMat;

  vector<unsigned short> newRmap=rmap;

  vector<unsigned short> distinct(inMat.rowsize());
  unsigned short iR;
  std::vector<vector<unsigned short> > RemainingTest(2);
  RemainingTest[0].resize(inMat.rowsize());   RemainingTest[1].resize(inMat.rowsize());
  unsigned short distinctSize=0, remainingSize=0, needToTestSize=inMat.rowsize();
  bool parity=0, newparity;

  if(inMat.rowsize() > 1){
    RemainingTest[0]=table((unsigned short) 0,(unsigned short) (inMat.rowsize()-1) );
    while(needToTestSize!=0){
      distinct[distinctSize]=RemainingTest[parity][0];
      distinctSize++;
      newparity=(parity+1)%2;
      remainingSize=0;
      for(iR=1; iR < needToTestSize; iR++){
	// not the same as row=needToTest[0]
	if(inMat[RemainingTest[parity][iR]]!=inMat[RemainingTest[parity][0]]){
	  //	if(sameHapQ(inMat,RemainingTest[parity][iR],RemainingTest[parity][0])==false){ 
	  RemainingTest[newparity][remainingSize]=RemainingTest[parity][iR];
	  remainingSize++;
	}
      }
      needToTestSize=remainingSize;
      parity=newparity;
    }

    reducedMat.mat.resize(distinctSize);
    for(iR=0; iR < distinctSize ; iR++){
      reducedMat.mat[iR]=inMat[distinct[iR]];
      rmap[iR]=newRmap[distinct[iR]];
    }  
    rmap.resize(distinctSize);
    newRmap.clear();
    RemainingTest.clear();
    distinct.clear();
    return(reducedMat);
  }
  else return(inMat);
}
      
// Remove a row and coalesce indentical sequences
BinMatrix coalesce(const unsigned short row2Remove, const BinMatrix &inMat, vector <unsigned short> &rmap){

  BinMatrix reducedMat;

  vector<unsigned short> newRmap=rmap;
  vector<unsigned short> distinct(inMat.rowsize());
  unsigned short iR;
  std::vector<vector<unsigned short> > RemainingTest(2);
  RemainingTest[0].resize(inMat.rowsize()-1);   RemainingTest[1].resize(inMat.rowsize()-1);
  unsigned short distinctSize=0, remainingSize=0, needToTestSize=inMat.rowsize()-1;
  bool parity=0, newparity;

  if(inMat.rowsize() > 1){
    RemainingTest[0]=brokenTable(0,(unsigned short) (inMat.rowsize()-1),row2Remove);
    while(needToTestSize!=0){
      distinct[distinctSize]=RemainingTest[parity][0];
      distinctSize++;
      newparity=(parity+1)%2;
      remainingSize=0;      
      for(iR=1; iR < needToTestSize; iR++){
	// not the same as row=needToTest[0]
	if(inMat[RemainingTest[parity][iR]]!=inMat[RemainingTest[parity][0]]){
	//	if(sameHapQ(inMat,RemainingTest[parity][iR],RemainingTest[parity][0])==false){ 
	  RemainingTest[newparity][remainingSize]=RemainingTest[parity][iR];
	  remainingSize++;
	}
      }
      needToTestSize=remainingSize;
      parity=newparity;
    }

    reducedMat.mat.resize(distinctSize);
    for(iR=0; iR < distinctSize ; iR++){
      reducedMat.mat[iR]=inMat[distinct[iR]];
      rmap[iR]=newRmap[distinct[iR]];
    }
    rmap.resize(distinctSize);
    newRmap.clear();
    RemainingTest.clear();
    distinct.clear();
    return(reducedMat);
  }
  else return(inMat);
}

unsigned int maxOfVec(const vector<unsigned int> &vec){
  unsigned int iN;
  unsigned int tmpMax=0;
  for(iN=0; iN < vec.size(); iN++){
    if(tmpMax < vec[iN]) tmpMax=vec[iN];
  }
  return tmpMax;
}


// collapse consecutive indentical columns
BinMatrix collapseCol(const BinMatrix &inMat){

  BinMatrix reducedMat;
  //  reducedMat.rowMap=inMat.rowMap;
  unsigned int countKept=1; // counts the number of columns to be kept
                            // always keep the first column
  vector<unsigned int> keep(inMat.colsize());
  unsigned int iC,iR;

  if(inMat.colsize() > 1){
    keep[0]=0;  // always keep the first column
    for(iC=1; iC < inMat.colsize(); iC++){
      if(inMat.column(iC)!=inMat.column(iC-1) &&
	 inMat.column(iC)!=conjugate(inMat.column(iC-1))){
	keep[countKept]=iC;
	countKept++;
      }
    }

    reducedMat.resize(inMat.rowsize(),countKept);
    for(iC=0; iC < countKept; iC++){
      for(iR=0; iR < inMat.rowsize(); iR++){
	reducedMat.mat[iR][iC]=inMat[iR][keep[iC]];
      }
    }
    return(reducedMat);
  }
  else return(inMat);
}
      
// coalesce, collapse and remove mutations interatively
BinMatrix reduceData(const BinMatrix & inMat, vector<unsigned short> &rmap)
{
  BinMatrix tempMat=inMat;
  unsigned int RowLength=0, ColLength=0;
  while(RowLength != tempMat.rowsize() || 
	ColLength != tempMat.colsize()){
    RowLength=tempMat.rowsize();
    ColLength=tempMat.colsize();
    tempMat=collapseCol(killNonInf(coalesce(tempMat,rmap)));
  }
  return(tempMat);
}

// Remove a row, and then coalesce, collapse and remove mutations interatively
BinMatrix removeAndReduce(const unsigned short row2Remove, const BinMatrix & inMat, vector<unsigned short> &rmap){
  BinMatrix tempMat=inMat;
  unsigned int RowLength=tempMat.rowsize()-1, ColLength=tempMat.colsize();
  // RowLength is tempMat.rowsize()-1 because we will remove a row from tempMat
  tempMat=collapseCol(killNonInf(coalesce(row2Remove,tempMat,rmap)));

  while(RowLength != tempMat.rowsize() || 
        ColLength != tempMat.colsize()){
    RowLength=tempMat.rowsize();
    ColLength=tempMat.colsize();
    tempMat=collapseCol(killNonInf(coalesce(tempMat,rmap)));
  }
  return(tempMat);
}

// get reduced sub-matrix between head and tail
BinMatrix getReducedSubMat(const BinMatrix & inMat, const unsigned int head, const unsigned int tail){
  BinMatrix tempMat(inMat.rowsize(),tail-head+1);
  unsigned int irow;
  for(irow=0; irow < inMat.rowsize(); irow++){
    std::copy(inMat.mat[irow].begin()+head,inMat.mat[irow].begin()+tail+1, tempMat.mat[irow].begin());
    //for(icol=head; icol <= tail; icol++){
    //      tempMat.mat[irow][icol-head]=inMat[irow][icol];
    //    }
  }
  unsigned int RowLength=0, ColLength=0;
  
  while(RowLength != tempMat.rowsize() || 
        ColLength != tempMat.colsize()){
    RowLength=tempMat.rowsize();
    ColLength=tempMat.colsize();
    tempMat=collapseCol(killNonInf(coalesceFL(tempMat)));
  }
  return(tempMat);
}  

// remove a row from matrix and return the modified matrix
// input matrix does not get changed
BinMatrix removeRow( unsigned int row2Remove, const BinMatrix & inMat, vector<unsigned short> &rmap){
  BinMatrix tempMat(inMat.rowsize()-1);
  unsigned int iR;
  unsigned int keptCount=0;
  vector <unsigned short> newmap=rmap;
  for(iR=0; iR < row2Remove; iR++){
    tempMat.mat[keptCount]=inMat[iR];
    rmap[keptCount]=newmap[iR];
    keptCount++;
  }
  for(iR=row2Remove+1; iR < inMat.rowsize(); iR++){
    tempMat.mat[keptCount]=inMat[iR];
    rmap[keptCount]=newmap[iR];
    keptCount++;
  }
  rmap.resize(rmap.size()-1);
  newmap.clear();
  return(tempMat);
}  

// complement of X in the n-set {1,2,...,n}
vector<unsigned short> complement(const vector<unsigned short> &inVec, unsigned short iN){
  unsigned short iR,iI;
  unsigned short keptCount=0;
  vector<unsigned short> tempVec(iN-inVec.size());
  if(inVec.size()!=0){
    for(iR=0; iR < inVec[0]; iR++){
      tempVec[keptCount]=iR;
      keptCount++;
    }
    for(iI=1; iI < inVec.size(); iI++){
      for(iR=inVec[iI-1]+1; iR < inVec[iI]; iR++){
	tempVec[keptCount]=iR;
	keptCount++;
      }
    }
    for(iR=inVec[inVec.size()-1]+1; iR < iN; iR++){
      tempVec[keptCount]=iR;
      keptCount++;
    }
    return(tempVec);
  }
  else return( table((unsigned short) 0,(unsigned short) (iN-1) ) );
} 

/*
vector<unsigned long>powerOf2(const unsigned int length){
  vector<unsigned long> result(length);
  for(unsigned int iN=0; iN < length; iN++){
    result[iN]=(unsigned long) pow( (double) 2, (int) iN);
  }
  return  result;
}

unsigned long map2long(const vector<unsigned short>& inVec){
  extern vector<unsigned long> powerTable;
  unsigned long result=0;
  for(unsigned int iN=0; iN < inVec.size(); iN++){
    result+=powerTable[inVec[iN]];
  }
  return result;
}
*/

// converts a vector of type vector<unsigned short> into a 2 bit vector
// encoding positions  
// e.g  vec=(2,4,5) --> (0,1,0,1,1,0,...,0)
vector<bool> convert2BoolVec(const vector<unsigned short> & inVec, const unsigned short length){
  vector<bool> tmpVec(length);
  unsigned short iN;
  for(iN=0; iN < length; iN++){
    tmpVec[iN]=0;
  }
  for(iN=0; iN < inVec.size(); iN++){
    tmpVec[inVec[iN]]=1;
  }
  return tmpVec;
}

vector<unsigned short> randomlySelect(const vector<unsigned short> &inVec, const unsigned short truncateLength){
  vector<unsigned short> reducedVec=table((unsigned short) 0,(unsigned short) inVec.size()-1);
  vector<unsigned short> outVec(truncateLength);
  unsigned short iP,iI;
  unsigned short randomValue;
  // unsigned short tmpValue;
  // vector<unsigned short> permVec=table((unsigned short) 0,(unsigned short) inVec.size()-1);


  if(inVec.size()==1){
    return inVec;
  }
  else{
    /*  Random Permutation.
    for(iP=1; iP < inVec.size(); iP++){
     randomValue=UDRandom(iP+1);
      if(randomValue != iP){
	tmpValue=permVec[iP];
	permVec[iP]=permVec[randomValue];
	permVec[randomValue]=tmpValue;
      }
    }
    for(iP=0; iP < truncateLength; iP++){
      outVec[iP]=inVec[permVec[iP]];
    }
    */

    for(iP=0; iP < truncateLength; iP++){
      randomValue=UDRandom(inVec.size()-iP);
      outVec[iP]=inVec[reducedVec[randomValue]];
      for(iI=randomValue; iI < inVec.size()-iP; iI++){
	reducedVec[iI]=reducedVec[iI+1];
      }
    }
    return outVec;
  }

  /*
  outVec=inVec;
  outVec.resize(truncateLength);
  return outVec;
  */
}


//  Random number generator
//    Taken from Bob Griffiths's program recomb58
//


extern long randomseed;

#define RAND01 (uni01(&randomseed))
/* Long period random number generator from
   Marsagalia, G. and Zaman, A.
	 "A new class of random number generators",
	 Annals of Applied Probability, 1, 462-480.
	 Possibilities for base b=2^24 generators are
	 r     s
	 39    25
	 28     8
	 25    11
	 24    10
	 These generators have VERY long cycles.
*/

#define b 16777216
#define xb 16777216.0
#define r 39
#define s 25
#define WARMUP 1000
long itable[r];
long c__=0L;

double uni01(long *irandomseed) {
	long z=1L,k;
	static int position=0;
	int seed,i,l;
	/**************  Initialize generator ************************/

	if(*irandomseed < 0) {
	/* Fill up itable with system random numbers, quick and dirty */
		seed= -(int)(*irandomseed);
		srand(seed);
		for(i=0;i<r;i++) {
			itable[i]=((long)rand())%b;
			if(itable[i]<=0) itable[i] *= (-1);
		}
		/* Make a random permutation of initial itable values */
		for(i=0;i<r;i++) {
			l=i+rand()%(r-i);
			/* Swap itable[l] and itable [i] */
			k=itable[l];itable[l]=itable[i];itable[i]=k;
		}
		/* Warmup main generator */
		for(i=0;i<WARMUP;i++) uni01(&z);
	}

	/************  Main code for random numbers ****************/

	/* Position is where the rth most recently generated number is in
		the array. Its a "wrap around array".
	*/
	if(position-s<0) z=itable[position+r-s]-itable[position]-c__ ;
	else z=itable[position-s]-itable[position]-c__ ;
	if(z>=0) c__=0L;
	else { z += b;c__ = 1L;}
	itable[position]=*irandomseed=z;
	/* Next oldest is to the right */
	position++;
	if(position==r) position=0;
	/* Really return from 0.5/2^24 to 1.0 - 0.5/2^24 */
	return (z+0.5)/xb;
}

#undef b
#undef xb
#undef r
#undef s
#undef WARMUP


// Uniform Discrete.  
// returns a number between 0 and size-1
unsigned short UDRandom(unsigned short size){
  //  Uniform Uran;
  return  (size==0? 0 : (unsigned short) floor(size*RAND01));
}

void ReadSeed(){
  ifstream inseed("randomseed", ios::in); 
  if(!inseed){
    cerr << "Warning:  'randomseed' file does not exist.\n";
    cerr << "         The default seed will be used for this simulation.\n\n";
  }
  else{
    inseed >> randomseed;
  }
  inseed.close();
}
   
void WriteSeed(){
  //  Uniform Uran;
  srand(randomseed);
  ofstream outseed("randomseed", ios::out);
  outseed << -rand();
  outseed.close();
}

